/*
 * crcSemiCPU.c
 *
 * Function to perform CRC Semi CPU mode.
 *
 * Copyright (C) 2016 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "crcSemiCPU.h"
#include <stdio.h>

extern g_crcSemiCpuCtrl gMyCrcCtrl;

void crcSemiCpuMode(crcBASE_t *crc, g_crcSemiCpuCtrl * crcSemiCpuCtrl)
{
#ifdef crcREG2
	if ( (crc != crcREG1) && (crc != crcREG2) )
	{
		/* Fault */
	}
#elif defined(crcREG1)
	if (crc != crcREG1)
	{
		/* Fault */
	}
#else
	if (crc != crcREG)
	{
		/* Fault */
	}
#endif
	else if (crcSemiCpuCtrl == NULL)
	{
		/* Fault */
	}
	else if (crcSemiCpuCtrl->g_crcTable == NULL)
	{
		/* Fault */
	}
	else
	{
		crcConfig_t sCrcParams;
		g_dmaCTRL   g_dmaCTRLPKT;

		if(crcSemiCpuCtrl->bInitialized == false)
		{
			crcSemiCpuCtrl->bInitialized = true;
			crcSemiCpuCtrl->i32LastEntry = 0l;
		}
		else
		{
			uint64 u64Signature = 0ull;
			u64Signature = crcGetSectorSig(crc, CRC_CH1);

			/* Check CRC */
			if (u64Signature != crcSemiCpuCtrl->g_crcTable->recs[crcSemiCpuCtrl->i32LastEntry].crc_value)
			{
				/* Fault */
				//printf("Wrong CRC in SW Module @ address: 0x%08X\n", crcTable->recs[i32LastEntry-1].addr);
				printf("F\n");
			}
			else
			{
				/* Pass */
				printf("P\n");
			}

			crcSemiCpuCtrl->i32LastEntry++;

			if (crcSemiCpuCtrl->i32LastEntry >= crcSemiCpuCtrl->g_crcTable->num_recs)
			{
				crcSemiCpuCtrl->i32LastEntry = 0l;
			}
		}

		while(crcSemiCpuCtrl->g_crcTable->recs[crcSemiCpuCtrl->i32LastEntry].size == 0ul)
		{
			if (crcSemiCpuCtrl->i32LastEntry >= crcSemiCpuCtrl->g_crcTable->num_recs)
			{
				crcSemiCpuCtrl->i32LastEntry = 0l;
			}
			else
			{
				crcSemiCpuCtrl->i32LastEntry++;
			}
		}

		/* Configure DMA Control Packed (structure is part of dma.c) */
		g_dmaCTRLPKT.SADD      = crcSemiCpuCtrl->g_crcTable->recs[crcSemiCpuCtrl->i32LastEntry].addr;  /* initial source address */
		g_dmaCTRLPKT.DADD      = (uint32_t)(&(crc->PSA_SIGREGL1)); /* initial destination address */
		g_dmaCTRLPKT.CHCTRL    = 0ul; /* channel control */
		g_dmaCTRLPKT.RDSIZE    = ACCESS_64_BIT; /* read size */
		g_dmaCTRLPKT.WRSIZE    = ACCESS_64_BIT; /* write size */
		g_dmaCTRLPKT.FRCNT     = (crcSemiCpuCtrl->g_crcTable->recs[crcSemiCpuCtrl->i32LastEntry].size + 7ul) / 8ul; /* frame count */
		g_dmaCTRLPKT.ELCNT     = 1;   /* element count */
		g_dmaCTRLPKT.ELSOFFSET = 0ul << g_dmaCTRLPKT.RDSIZE; /* element source offset */
		g_dmaCTRLPKT.FRSOFFSET = 0ul << g_dmaCTRLPKT.RDSIZE; /* frame source offset */
		g_dmaCTRLPKT.ELDOFFSET = 0ul << g_dmaCTRLPKT.WRSIZE; /* element destination offset */
		g_dmaCTRLPKT.FRDOFFSET = 0ul << g_dmaCTRLPKT.WRSIZE; /* frame destination offset */
#ifdef RM57_OR_LC43
		g_dmaCTRLPKT.PORTASGN  = 1ul; /* Port A Read (Flash) - Port B Write (CRC) */
#else
		g_dmaCTRLPKT.PORTASGN  = 4ul; /* only Port B */
#endif
		g_dmaCTRLPKT.TTYPE     = FRAME_TRANSFER ; /* transfer type */
		g_dmaCTRLPKT.ADDMODERD = ADDR_INC1;     /* address mode read */
		g_dmaCTRLPKT.ADDMODEWR = ADDR_FIXED;     /* address mode write */
		g_dmaCTRLPKT.AUTOINIT  = AUTOINIT_OFF;     /* autoinit off */

		sCrcParams.crc_channel   = CRC_CH1;
		sCrcParams.mode          = CRC_SEMI_CPU;
		sCrcParams.pcount        = g_dmaCTRLPKT.FRCNT;
		sCrcParams.scount        = 1u;
		sCrcParams.wdg_preload   = 0u;
		sCrcParams.block_preload = (g_dmaCTRLPKT.FRCNT * 10ul * 10ul * 2ul) / 64ul; /* The timeout counter is clocked by a prescaler clock which is permanently running at division 64 of HCLK clock. */

		crcChannelReset(crc, CRC_CH1);

		printf("Checking CRC for address 0x%08X, %d elements\r\n", g_dmaCTRLPKT.SADD, g_dmaCTRLPKT.FRCNT);

		crcSetConfig(crc, &sCrcParams);

		/* Assign DMA Control Packet to DMA Channel */
		dmaSetCtrlPacket(crcSemiCpuCtrl->eDmaChannel, g_dmaCTRLPKT);

		dmaReqAssign(crcSemiCpuCtrl->eDmaChannel, crcSemiCpuCtrl->u32DmaRequLine);

		/* Set the DMA Channel to trigger on h/w request */
		dmaSetChEnable(crcSemiCpuCtrl->eDmaChannel, DMA_HW);
	}

	return;
}

void crcNotification(crcBASE_t *crc, uint32 flags)
{
	if (CRC_CH1_CC == flags)
	{
		crcSemiCpuMode(crcREG, &gMyCrcCtrl);
	}
	else
	{
		_disable_IRQ();
#ifdef RM57_OR_LC43
		rtiDisableNotification(rtiREG1, 1ul << 11); /* Disable DMA request 3 (bit 11) */
#else
		rtiDisableNotification(1ul << 11); /* Disable DMA request 3 (bit 11) */
#endif
		crcDisableNotification(crc,
			CRC_CH1_CC |
			CRC_CH1_FAIL |
			CRC_CH1_OR |
			CRC_CH1_UR |
			CRC_CH1_TO);

		printf("Fault in crcNotification, flags: 0x%08X\r\n", flags);
	}

	return;
}
